<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\User;
use App\Models\Order;
use App\Models\Customer;
use DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    public function billNotUpdated(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $paymentMode = $request->input('payment_mode');

        // Fetch data
        $transactions = DB::table('orders')
            ->join('order_receipt', 'orders.id', '=', 'order_receipt.order_id')
            ->join('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->join('customers', 'customers.id', '=', 'order_receipt.customer_name')
            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->where('orders.payment_status', 'Partially Paid')
             ->where('orders.bill_generate', 0)
            ->whereBetween('order_transaction.transaction_date', [$startDate, $endDate])
            ->when($paymentMode, function ($query, $paymentMode) {
                $query->where('order_transaction.payment_mode', $paymentMode);
            })
            ->select(
                'orders.id as order_id',
                'orders.order_code',
                'orders.payment_status',

                'order_receipt.id as receipt_id',
                'order_receipt.receipt_type',
                 'order_receipt.receipt_id as rec_no',
                'order_receipt.customer_name as customer_id',
                'order_transaction.transaction_date',
                'order_transaction.order_amount',
                'order_transaction.pay_amount',         // Actual paid in that transaction
                'order_transaction.amount_paid',        // Total paid till that point
                'order_transaction.remaining_amount',
                'order_transaction.payment_mode',
                'order_transaction.cheque_number',
                'order_transaction.reference_id',
                'order_transaction.bank_name',
                'order_transaction.ifsc_code',
                'order_transaction.account_number',
                'order_transaction.remark',

                'banks.bank_name',
                'customers.customer_name'
            )
            ->orderBy('orders.id')
            ->orderBy('order_transaction.id') // ensure chronological
            ->get();

        // Group by order_id
        $groupedOrders = $transactions->groupBy('order_id');

        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // dd($groupedOrders);

        return view('reportstemplate.billnotupdated', [
            'groupedOrders' => $groupedOrders,
            'company' => $company,
            'is_pdf' => false
        ]);
    }

    public function billNotUpdatedpdf(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $paymentMode = $request->input('payment_mode');

        // Fetch data same as billNotUpdated()
        $transactions = DB::table('orders')
            ->join('order_receipt', 'orders.id', '=', 'order_receipt.order_id')
            ->join('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->join('customers', 'customers.id', '=', 'order_receipt.customer_name')
            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->where('orders.payment_status', 'Partially Paid')
             ->where('orders.bill_generate', 0)
            ->whereBetween('order_transaction.transaction_date', [$startDate, $endDate])
            ->when($paymentMode, function ($query, $paymentMode) {
                $query->where('order_transaction.payment_mode', $paymentMode);
            })
            ->select(
                'orders.id as order_id',
                'orders.order_code',
                'orders.payment_status',
                'order_receipt.receipt_id as rec_no',
                'order_receipt.id as receipt_id',
                'order_receipt.receipt_type',
                'order_receipt.customer_name as customer_id',

                'order_transaction.transaction_date',
                'order_transaction.order_amount',
                'order_transaction.pay_amount',
                'order_transaction.amount_paid',
                'order_transaction.remaining_amount',
                'order_transaction.payment_mode',
                'order_transaction.cheque_number',
                'order_transaction.reference_id',
                'order_transaction.bank_name',
                'order_transaction.ifsc_code',
                'order_transaction.account_number',
                'order_transaction.remark',

                'banks.bank_name',
                'customers.customer_name'
            )
            ->orderBy('orders.id')
            ->orderBy('order_transaction.id')
            ->get();

        // Group by order_id
        $groupedOrders = $transactions->groupBy('order_id');

        // Fetch company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // Generate PDF from the same view
        $pdf = PDF::loadView('reportstemplate.billnotupdated', [
            'groupedOrders' => $groupedOrders,
            'company' => $company,
            'is_pdf' => true
        ]); // optional: landscape

        return $pdf->download('bill_report.pdf');
    }




    public function creditReport(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate   = $request->input('end_date') ?? date('Y-m-d');
            $customer  = $request->input('customer');

            $customers = DB::table('customers')->whereNull('deleted_at')->get();

            // Get ALL transactions, filter by order_date instead
            $transactions = DB::table('orders')
                ->join('customers', 'orders.customer_id', '=', 'customers.id')
                ->leftJoin('order_receipt', 'orders.id', '=', 'order_receipt.order_id')
                ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                ->select(
                    'orders.id as order_id',
                    'orders.order_code',
                    'orders.order_date',
                    'orders.payment_status',

                    'customers.id as customer_id',
                    'customers.customer_name',
                    'customers.mobile',
                     'order_receipt.receipt_id as rec_no',
                    'order_transaction.transaction_date',
                    'order_transaction.payment_mode',
                    'order_transaction.pay_amount',
                    'order_transaction.amount_paid',
                    'order_transaction.remaining_amount',
                    'order_transaction.bank_name',
                    'order_transaction.account_number',
                    'order_transaction.ifsc_code',
                    'order_transaction.cheque_number',
                    'order_transaction.reference_id'
                )
                ->whereIn('orders.payment_status', ['Partially Paid'])
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('orders.order_date', [$startDate, $endDate]);
                })
                ->when($customer, fn($q) => $q->where('orders.customer_id', $customer))
                ->orderBy('customers.customer_name')
                ->orderBy('orders.id')
                ->orderBy('order_transaction.id')
                ->get();

            $grouped = $transactions->groupBy(function ($item) {
                return $item->customer_id . '-' . $item->order_id;
            });

            

            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            return view('reportstemplate.creditreport', [
                'grouped' => $grouped,
                'company' => $company,
                'customers' => $customers,
                'is_pdf' => false
            ]);
        }

    public function creditReportpdf(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate   = $request->input('end_date') ?? date('Y-m-d');
            $customer  = $request->input('customer');

            // Fetch all relevant transactions
            $transactions = DB::table('orders')
                ->join('customers', 'orders.customer_id', '=', 'customers.id')
                ->leftJoin('order_receipt', 'orders.id', '=', 'order_receipt.order_id')
                ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                ->select(
                    'orders.id as order_id',
                    'orders.order_code',
                    'orders.order_date',
                    'orders.payment_status',
                     'order_receipt.receipt_id as rec_no',
                    'customers.id as customer_id',
                    'customers.customer_name',
                    'customers.mobile',

                    'order_transaction.transaction_date',
                    'order_transaction.payment_mode',
                    'order_transaction.pay_amount',
                    'order_transaction.amount_paid',
                    'order_transaction.remaining_amount',
                    'order_transaction.bank_name',
                    'order_transaction.account_number',
                    'order_transaction.ifsc_code',
                    'order_transaction.cheque_number',
                    'order_transaction.reference_id'
                )
                ->whereIn('orders.payment_status', ['Partially Paid'])
                ->when($startDate && $endDate, fn($query) => $query->whereBetween('orders.order_date', [$startDate, $endDate]))
                ->when($customer, fn($q) => $q->where('orders.customer_id', $customer))
                ->orderBy('customers.customer_name')
                ->orderBy('orders.id')
                ->orderBy('order_transaction.id')
                ->get();

            // Group by customer+order
            $grouped = $transactions->groupBy(function ($item) {
                return $item->customer_id . '-' . $item->order_id;
            });

            // Calculate grand balance (sum of latest remaining amounts)
            $grandBalance = $transactions
                ->groupBy('order_id')
                ->map(fn($items) => $items->last()->remaining_amount ?? 0)
                ->sum();

            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            // Generate PDF using same view
            $pdf = PDF::loadView('reportstemplate.creditreport', [
                'grouped' => $grouped,
                'company' => $company,
                'is_pdf' => true,
                'grandBalance' => $grandBalance,
                'customers' => [] // Optional
            ]);

            return $pdf->download('credit_report.pdf');
        }


   
    public function customerReceiptDetails(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        $customerId = $request->input('customer');
         $payment_mode = $request->input('payment_mode');
         $order = $request->input('order');

        // Setup base query
        $query = DB::table('order_receipt')
            ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
            ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name') // customer_name is actually customer_id
            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->select(
                'order_receipt.*',
                'orders.order_code',
                'banks.bank_name as bank_name',
                'customers.customer_name as customer_display_name',
                'order_receipt.customer_name as customer_id',
                'order_transaction.payment_mode',
                'order_transaction.pay_amount',
                'order_transaction.cheque_number',
                'order_transaction.transaction_date as trans_date',
                'order_transaction.ifsc_code',
                'order_transaction.account_number',
                'order_transaction.reference_id',
                'order_transaction.remark'
            );

        // Apply filters only when needed
        if ($startDate && $endDate) {
            $query->whereBetween('order_transaction.transaction_date', [
                $startDate . ' 00:00:00',
                $endDate . ' 23:59:59',
            ]);
        } elseif (!$customerId) {
            // Only apply today's filter if no customer and no date
            $today = date('Y-m-d');
            $query->whereDate('order_transaction.transaction_date', $today);
        }

        if (!empty($customerId)) {
            $query->where('order_receipt.customer_name', $customerId);
        }

         if (!empty($order)) {
                        $query->where('order_transaction.order_id', $order);
                    } 
        
        if (!empty($payment_mode)) {
            $query->where('order_transaction.payment_mode', $payment_mode);
        }

        $receipts = $query->orderBy('order_transaction.id', 'desc')->get();

        $groupedReceipts = $receipts->groupBy(function ($receipt) {
            return Carbon::parse($receipt->trans_date)->format('Y-m-d');
        })->map(function ($dateGroup) {
            return $dateGroup->groupBy('customer_id');
        });

        $customers = DB::table('customers')->whereNull('deleted_at')->get();
        $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();


        return view('reportstemplate.customerreceiptdetails', [
            'groupedReceipts' => $groupedReceipts,
            'customers' => $customers,
             'orders'=>$orders,
            'is_pdf' => false
        ]);
    }


    public function customerReceiptDetailspdf(Request $request)
    {
        // Safely handle large data
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        // Request inputs
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        $customerId = $request->input('customer');
         $payment_mode = $request->input('payment_mode');
             $order = $request->input('order');

        // Build query
        $query = DB::table('order_receipt')
            ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
            ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name') // customer_name is customer_id
            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->select(
                'order_receipt.*',
                'orders.order_code',
                'banks.bank_name as bank_name',
                'customers.customer_name as customer_display_name',
                'order_receipt.customer_name as customer_id',
                'order_transaction.payment_mode',
                'order_transaction.pay_amount',
                'order_transaction.cheque_number',
                'order_transaction.transaction_date as trans_date',
                'order_transaction.ifsc_code',
                'order_transaction.account_number',
                'order_transaction.reference_id',
                'order_transaction.remark'
            );

        // Apply filters
        if ($startDate && $endDate) {
            $query->whereBetween('order_transaction.transaction_date', [
                $startDate . ' 00:00:00',
                $endDate . ' 23:59:59',
            ]);
        } elseif (!$customerId) {
            // If no filter at all, use today's data
            $today = date('Y-m-d');
            $query->whereDate('order_transaction.transaction_date', $today);
        }

        if (!empty($customerId)) {
            $query->where('order_receipt.customer_name', $customerId);
        }

          if (!empty($order)) {
                        $query->where('order_transaction.order_id', $order);
                    } 
                    
            if (!empty($payment_mode)) {
                $query->where('order_transaction.payment_mode', $payment_mode);
            }

        // Execute query
        $receipts = $query->orderBy('order_transaction.id', 'desc')->get();

        // Group by transaction date -> then customer_id
        $groupedReceipts = $receipts->groupBy(function ($receipt) {
            return Carbon::parse($receipt->trans_date)->format('Y-m-d');
        })->map(function ($dateGroup) {
            return $dateGroup->groupBy('customer_id');
        });

         $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();
            $customers = DB::table('customers')->whereNull('deleted_at')->get();

        // Generate PDF
        $pdf = PDF::loadView('reportstemplate.customerreceiptdetails', [
            'groupedReceipts' => $groupedReceipts,
              'customers'=>$customers,
                 'orders'=>$orders,
            'is_pdf' => true
        ])->setPaper('a3', 'portrait');

        // File name
        $filename = 'customer_receipt_' . now()->format('d_M_Y') . '.pdf';

        return $pdf->download($filename);
    }

    public function daySales(Request $request){

            // Safely handle large data
            ini_set('memory_limit', '1024M'); // Increase memory limit
            set_time_limit(300);              // Increase max execution time to 5 minutes

            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');


            if (!$request->has('start_date') && !$request->has('end_date')) {
                $startDate = date('Y-m-d');
                $endDate = date('Y-m-d');
            }
            // dd($request->all(), $startDate, $endDate);
            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';

             $startDate = $request->input('start_date') ?: date('Y-m-d');
             $endDate = $request->input('end_date') ?: date('Y-m-d');

             $customer = $request->input('customer');
             $payment_mode = $request->input('payment_mode');
             $order = $request->input('order');

            $query = DB::table('order_receipt')
            ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
              ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                ->select(
                    'order_receipt.*',
                    'orders.order_code',
                    'banks.bank_name as bank_name',
                    'order_transaction.amount_paid',
                     'order_transaction.payment_mode',
                    'order_transaction.pay_amount',
                    'order_transaction.cheque_number',
                    'order_transaction.transaction_date as trans_date',
                    'order_transaction.ifsc_code',
                    'order_transaction.account_number',
                    'order_transaction.reference_id',
                    'order_transaction.remark',
                    'customers.customer_name'
                );
            // ->whereDate('order_receipt.created_at', today()) 
              //  ->whereBetween('order_receipt.created_at', [$startDateTime, $endDateTime]);
                if ($request->has(['start_date', 'end_date'])) {
                            $query->whereBetween('order_transaction.transaction_date', [
                                $startDate . ' 00:00:00',
                                $endDate . ' 23:59:59',
                            ]);
                        } else {
                            // Show today's data by default
                            $query->whereDate('order_transaction.transaction_date', $startDate);
                        }

                if (!empty($customer)) {
                $query->where('orders.customer_id', $customer);
                }

                 if (!empty($order)) {
                        $query->where('order_transaction.order_id', $order);
                    } 
                    
                    if (!empty($payment_mode)) {
                        $query->where('order_transaction.payment_mode', $payment_mode);
                    }

            $order_receipts = $query->orderBy('customers.customer_name')->orderBy('order_transaction.id','desc')->get();
            
                // ->orderBy('payment_method') 
                // ->orderBy('customers.customer_name') 
                // ->orderBy('order_receipt.order_id') 
                // ->get();
            //dd($order_receipts);
            $grouped_data = [];

            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode;
                $customer = $receipt->customer_name;

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer])) {
                    $grouped_data[$payment_method][$customer] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer]['orders'][] = [
                    'order_id' => $receipt->order_id,
                    'order_code' => $receipt->order_code,
                    'amount' => number_format($receipt->pay_amount, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'ifsc_code' => $receipt->ifsc_code,
                    'account_number' => $receipt->account_number,
                    'reference_id' => $receipt->reference_id,
                    'transaction_date' => $receipt->transaction_date,
                    'remark' => $receipt->remark,
                    'cheq_no' => $receipt->cheque_number,
                    'rcpt_type' => $receipt->receipt_type
                ];

                $grouped_data[$payment_method][$customer]['total_amount'] += $receipt->pay_amount;
            }

            // 3. Fetch Kukreja Interiors company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
                ->first();

                $customers = DB::table('customers')->where('deleted_at',null)->get();
                $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();
            //dd($grouped_data);
            return view('reportstemplate.daysales', [
                'grouped_data' => $grouped_data,
                'company'=>$company,
                'customers'=>$customers,
                'orders'=>$orders,
                'is_pdf' => false // Not a PDF
                ]);

    }

    public function daySalespdf(Request $request)
        {
            // Safely handle large data
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');

            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';

            $customer = $request->input('customer');
             $payment_mode = $request->input('payment_mode');
            $order = $request->input('order');

            $query = DB::table('order_receipt')
                ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
                ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
                ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                ->select(
                    'order_receipt.*',
                    'orders.order_code',
                    'banks.bank_name as bank_name',
                    'order_transaction.amount_paid',
                    'order_transaction.payment_mode',
                    'order_transaction.pay_amount',
                    'order_transaction.cheque_number',
                    'order_transaction.transaction_date as transaction_date',
                    'order_transaction.ifsc_code',
                    'order_transaction.account_number',
                    'order_transaction.reference_id',
                    'order_transaction.remark',
                    'customers.customer_name'
                )
                ->whereBetween('order_receipt.created_at', [$startDateTime, $endDateTime]);

            if (!empty($customer)) {
                $query->where('orders.customer_id', $customer);
            }

             if (!empty($order)) {
                        $query->where('order_transaction.order_id', $order);
                    } 
                    
            if (!empty($payment_mode)) {
                $query->where('order_transaction.payment_mode', $payment_mode);
            }

            $order_receipts = $query->orderBy('customers.customer_name')
                ->orderBy('order_transaction.id','desc')
                ->get();

            // Group by payment method and customer
            $grouped_data = [];

            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode ?? 'Unknown';
                $customer_name = $receipt->customer_name ?? 'Unknown';

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer_name])) {
                    $grouped_data[$payment_method][$customer_name] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer_name]['orders'][] = [
                    'order_id' => $receipt->order_id,
                    'order_code' => $receipt->order_code,
                    'amount' => number_format($receipt->pay_amount, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'ifsc_code' => $receipt->ifsc_code,
                    'account_number' => $receipt->account_number,
                    'reference_id' => $receipt->reference_id,
                    'transaction_date' => $receipt->transaction_date,
                    'remark' => $receipt->remark,
                    'cheq_no' => $receipt->cheque_number,
                    'rcpt_type' => $receipt->receipt_type
                ];

                $grouped_data[$payment_method][$customer_name]['total_amount'] += $receipt->pay_amount;
            }

            // Company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            // Customers (optional in case view uses it)
            $customers = DB::table('customers')->whereNull('deleted_at')->get();
              $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();
            // Generate PDF
            $pdf = PDF::loadView('reportstemplate.daysales', [
                'grouped_data' => $grouped_data,
                'company' => $company,
                'customers' => $customers,
                'orders' => $orders,
                'is_pdf' => true
            ])->setPaper('a3', 'portrait');

            return $pdf->download('day_sales_' . now()->format('d_M_Y') . '.pdf');
        }

    public function invSales(Request $request)
    {
        // Safely handle large data
        ini_set('memory_limit', '1024M'); // Increase memory limit
        set_time_limit(300);              // Increase max execution time to 5 minutes

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $customer = $request->input('customer');

        // Get all necessary order item data, joined with related tables
        $orderItemsQuery = DB::table('order_items')
            ->join('orders', 'orders.id', '=', 'order_items.order_id')
             ->leftJoin('invoice_nos','invoice_nos.order_id','=','orders.id')
            ->join('order_receipt', 'order_receipt.order_id', '=', 'orders.id')
            ->join('customers', 'customers.id', '=', 'orders.customer_id')
            ->join('products', 'products.id', '=', 'order_items.product_id')
            ->join('pricing', 'pricing.product_id', '=', 'products.id')
            ->select(
                'order_receipt.order_date as invoice_date',
                'orders.id as invoice_id',
                'invoice_nos.invoice_no',
                'customers.customer_name as customer_name',
                'customers.gst_number',
                'pricing.gst_rate',
                'orders.subtotal as base_amount',
                'products.hsn_code'
            )
            ->whereBetween('order_receipt.order_date', [$startDate, $endDate]);

        // Apply customer filter if selected
        if (!empty($customer)) {
            $orderItemsQuery->where('orders.customer_id', $customer);
        }

        $orderItems = $orderItemsQuery->get();

        // Step 2: Add calculated tax amounts
        foreach ($orderItems as $item) {
            $gstRate = $item->gst_rate;
            $base = $item->base_amount;

            // Calculate SGST and CGST as 50% of GST each (intra-state)
            $item->sgst_amount = round(($base * $gstRate) / 2 / 100, 2);
            $item->cgst_amount = round(($base * $gstRate) / 2 / 100, 2);
            $item->igst_amount = 0; // Update if IGST logic is needed
        }

        // Step 3: Group by GST rate
        $groupedData = [];
        foreach ($orderItems as $item) {
            $gstRate = $item->gst_rate;

            if (!isset($groupedData[$gstRate])) {
                $groupedData[$gstRate] = [
                    'percentage' => $gstRate,
                    'sales_items' => []
                ];
            }

            $groupedData[$gstRate]['sales_items'][] = $item;
        }

        // Fetch Kukreja Interiors company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        $customers = DB::table('customers')->whereNull('deleted_at')->get();

        $salesdata = array_values($groupedData); // Reset array keys
        $gst_summary = [];

        return view('reportstemplate.invsales', [
            'salesdata' => $salesdata,
            'gst_summary' => $gst_summary,
            'company' => $company,
            'customers' => $customers,
            'is_pdf' => false,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'selected_customer' => $customer,
        ]);
    }


    public function invSalespdf(Request $request){

        // Safely handle large data
        ini_set('memory_limit', '1024M'); // Increase memory limit
        set_time_limit(300);              // Increase max execution time to 5 minutes
      $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $customer = $request->input('customer');

        // Get all necessary order item data, joined with related tables
        $orderItems = DB::table('order_items')
            ->join('orders', 'orders.id', '=', 'order_items.order_id')
             ->leftJoin('invoice_nos','invoice_nos.order_id','=','orders.id')   
            ->join('order_receipt', 'order_receipt.order_id', '=', 'orders.id')
            ->join('customers', 'customers.id', '=', 'orders.customer_id')
            ->join('products', 'products.id', '=', 'order_items.product_id')
            ->join('pricing', 'pricing.product_id', '=', 'products.id')
            ->select(
                'order_receipt.order_date as invoice_date',
                'orders.id as invoice_id',
                'customers.customer_name as customer_name',
                'customers.gst_number',
                'pricing.gst_rate',
                'invoice_nos.invoice_no',
                // 'pricing.basic_cost as base_amount',
                'orders.subtotal as base_amount',
                'products.hsn_code'
            )
             ->whereBetween('order_receipt.order_date', [$startDate, $endDate]);

        // Apply customer filter if selected
        if (!empty($customer)) {
            $orderItems->where('orders.customer_id', $customer);
        }

        $orderItems = $orderItems->get();



            // Step 2: Add calculated tax amounts
            foreach ($orderItems as $item) {
                $gstRate = $item->gst_rate;
                $base = $item->base_amount;

                // Calculate SGST and CGST as 50% of GST each (intra-state)
                $item->sgst_amount = round(($base * $gstRate) / 2 / 100, 2);
                $item->cgst_amount = round(($base * $gstRate) / 2 / 100, 2);
                $item->igst_amount = 0; // Update if IGST logic is needed
            }

            // Step 3: Group by GST rate
            $groupedData = [];
            foreach ($orderItems as $item) {
                $gstRate = $item->gst_rate;

                if (!isset($groupedData[$gstRate])) {
                    $groupedData[$gstRate] = [
                        'percentage' => $gstRate,
                        'sales_items' => []
                    ];
                }

                $groupedData[$gstRate]['sales_items'][] = $item;
            }

            // 3. Fetch Kukreja Interiors company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
                ->first();

            $salesdata = array_values($groupedData); // Reset array keys
            $gst_summary = [];


        $pdf = PDF::loadView('reportstemplate.invsales', [
        'salesdata' => $salesdata,
        'gst_summary' => $gst_summary,
        'company'=>$company,
            'is_pdf' => true 
        ]);
        return $pdf->download('inv_sales.pdf'); 
    }


     public function labourWorkdowns(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            // ✅ Default values for today
            $startDate = $request->input('start_date', date('Y-m-d'));
            $endDate = $request->input('end_date', date('Y-m-d'));
            $customerId = $request->input('customer');
            $installerId = $request->input('installer');

            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
                ->first();

            $labourData = DB::table('order_items')
                ->join('orders', 'orders.id', '=', 'order_items.order_id')
                ->join('customers', 'customers.id', '=', 'orders.customer_id')
                ->leftJoin('installer_charges', function ($join) {
                        $join->on('installer_charges.user_id', '=', 'orders.installer_id')
                            ->whereColumn('installer_charges.product_id', 'order_items.product_id'); 
                        // Use whereColumn instead of on to avoid overriding the first condition
                    })
                ->join('users', 'users.id', '=', 'orders.installer_id')
                ->select(
                    'order_items.order_id',
                    'order_items.updated_at',
                    'customers.customer_name',
                    'order_items.product_id',
                    'order_items.quantity',
                    'order_items.price',
                    'orders.order_code',
                    'installer_charges.rate as labour_rate',
                    'installer_charges.user_id',
                    'users.full_name as labour_name'
                )
                ->whereBetween('orders.order_date', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']) // ✅ Default to today
                ->when($customerId, function ($query) use ($customerId) {
                    return $query->where('customers.id', $customerId);
                })
                ->when($installerId, function ($query) use ($installerId) {
                    return $query->where('orders.installer_id', $installerId);
                })
                ->orderBy('orders.id', 'desc')
                ->get();
            //    dd( $labourData);

            $customers = DB::table('customers')->where('deleted_at', null)->get();
            $installers = DB::table('users')->where('type_id', 5)->where('deleted_at', null)->get();

            $workdowns = [
                [
                    'labour' => 'All Work',
                    'workdown_items' => $labourData,
                ]
            ];

            return view('reportstemplate.labourworkdowns', [
                'workdowns' => $workdowns,
                'company'=>$company,
                'customers' => $customers,
                'installers' =>$installers,
                'is_pdf' => false,
                'startDate' => $startDate,
                'endDate' => $endDate
            ]);
        }



      public function labourWorkdownspdf(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            // ✅ Default values for today
            $startDate = $request->input('start_date', date('Y-m-d'));
            $endDate = $request->input('end_date', date('Y-m-d'));
            $customerId = $request->input('customer');
            $installerId = $request->input('installer');

            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
                ->first();

            $labourData = DB::table('order_items')
                ->join('orders', 'orders.id', '=', 'order_items.order_id')
                ->join('customers', 'customers.id', '=', 'orders.customer_id')
                ->leftJoin('installer_charges', function ($join) {
                    $join->on('installer_charges.user_id', '=', 'orders.installer_id')
                        ->on('installer_charges.product_id', '=', 'order_items.product_id');
                })
                ->join('users', 'users.id', '=', 'orders.installer_id')
                ->select(
                    'order_items.order_id',
                    'order_items.updated_at',
                    'customers.customer_name',
                    'order_items.product_id',
                    'order_items.quantity',
                    'order_items.price',
                    'orders.order_code',
                    'installer_charges.rate as labour_rate',
                    'installer_charges.installer_id',
                    'users.full_name as labour_name'
                )
                ->whereBetween('orders.order_date', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']) // ✅ Default to today
                ->when($customerId, function ($query) use ($customerId) {
                    return $query->where('customers.id', $customerId);
                })
                ->when($installerId, function ($query) use ($installerId) {
                    return $query->where('orders.installer_id', $installerId);
                })
                ->orderBy('orders.id', 'desc') // ✅ Latest orders first
                ->get();

            $workdowns = [
                [
                    'labour' => 'All Work',
                    'workdown_items' => $labourData,
                ]
            ];

            // ✅ Generate PDF with same view and variables
            $pdf = PDF::loadView('reportstemplate.labourworkdowns', [
                'workdowns' => $workdowns,
                'company' => $company,
                'is_pdf' => true,
                'startDate' => $startDate,
                'endDate' => $endDate
            ]);

            return $pdf->download('labour_workdowns_' . date('Ymd_His') . '.pdf');
        }


    // public function productSupplier(Request $request)
    // {
    //     // Safely handle large data
    //     ini_set('memory_limit', '1024M'); // Increase memory limit
    //     set_time_limit(300);              // Increase max execution time to 5 minutes

    //     // Get filter values
    //     $supplierId = $request->input('supplier');
    //     $productId = $request->input('product');
    //      $warehouseId = $request->input('warehouse');

    //     // Ensure proper types
    //     $supplierId = $supplierId ? (int) $supplierId : null;
    //     $productId = $productId ? (int) $productId : null;
    //      $warehouseId = $warehouseId ? (int) $warehouseId : null;


    //     // Fetch all suppliers
    //     $suppliers = DB::table('suppliers')
    //         ->select('id', 'supplier_name')
    //         ->whereNull('deleted_at')
    //         ->get();

    //      $warehouses = DB::table('warehouses')
    //                ->join('inventory','warehouses.id','=','inventory.warehouse_id')
    //                 ->select('warehouses.id', 'warehouses.warehouse_name')
    //                 ->whereNull('warehouses.deleted_at')
    //                 ->groupBy('inventory.warehouse_id')
    //                 ->get();


    //     // Product list will be fetched via AJAX — keep empty
    //     $products = collect();

    //     // Build base inward query with optional filters
    //     $inwardsQuery = DB::table('inventory')
    //         ->leftJoin('suppliers', 'suppliers.id', '=', 'inventory.supplier_id')
    //         ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
    //         ->select('inventory.*', 'suppliers.supplier_name', 'warehouses.warehouse_name');

    //     if ($supplierId) {
    //         $inwardsQuery->where('inventory.supplier_id', $supplierId);
    //     }

    //     if ($productId) {
    //         $inwardsQuery->where('inventory.product_id', $productId);
    //     }

    //      if ($warehouseId) {
    //                 $inwardsQuery->where('inventory.warehouse_id', $warehouseId);
    //             }

    //     $inwards = $inwardsQuery->get();

    //     // Gather rolls and related product/variant info
    //     $product_items = [];

    //     foreach ($inwards as $inward) {
    //         $rolls = DB::table('rolls')
    //             ->where('rolls.inward_id', $inward->id)
    //             ->leftJoin('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
    //             ->leftJoin('products', 'product_variant.product_id', '=', 'products.id')
    //             ->select(
    //                 'rolls.id',
    //                 'rolls.availability',
    //                 'products.product_name',
    //                 'product_variant.varient_code',
    //                 'product_variant.varient_name'
    //             )
    //             ->get();

    //         if (!$rolls->isEmpty()) {
    //             $product_items[] = [
    //                 'supplier_name' => $inward->supplier_name ?? '-',
    //                 'warehouse' => $inward->warehouse_name ?? '-',
    //                 'rolls' => $rolls
    //             ];
    //         }
    //     }

    //     // Company info
    //     $company = DB::table('companies')
    //         ->where('company_name', 'Kukreja Interiors')
    //         ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
    //         ->first();

    //     // Return view
    //     return view('reportstemplate.productsuppliermapping', [
    //         'product_items' => $product_items,
    //         'company' => $company,
    //         'suppliers' => $suppliers,
    //         'products' => $products, 
    //         'warehouses' => $warehouses, 
    //         'selected_supplier' => $supplierId,
    //         'selected_product' => $productId,
    //          'selected_warehouse' => $request->input('warehouse'),
    //         'is_pdf' => false
    //     ]);
    // }

    

    // public function productSupplierpdf(Request $request)
    // {
    //     // Handle large data
    //     ini_set('memory_limit', '1024M');
    //     set_time_limit(300);

    //     // Get filter values
    //     $supplierId = $request->input('supplier');
    //     $productId = $request->input('product');
    //     $warehouseId = $request->input('warehouse');

    //     // Ensure proper types
    //     $supplierId = $supplierId ? (int) $supplierId : null;
    //     $productId = $productId ? (int) $productId : null;
    //     $warehouseId = $warehouseId ? (int) $warehouseId : null;

    //     // Fetch all suppliers
    //     $suppliers = DB::table('suppliers')
    //         ->select('id', 'supplier_name')
    //         ->whereNull('deleted_at')
    //         ->get();

    //     // Fetch warehouses linked with inventory
    //     $warehouses = DB::table('warehouses')
    //         ->join('inventory', 'warehouses.id', '=', 'inventory.warehouse_id')
    //         ->select('warehouses.id', 'warehouses.warehouse_name')
    //         ->whereNull('warehouses.deleted_at')
    //         ->groupBy('inventory.warehouse_id')
    //         ->get();

    //     // Product list stays empty in PDF (for dropdown consistency)
    //     $products = collect();

    //     // Build query with filters
    //     $inwardsQuery = DB::table('inventory')
    //         ->leftJoin('suppliers', 'suppliers.id', '=', 'inventory.supplier_id')
    //         ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
    //         ->select('inventory.*', 'suppliers.supplier_name', 'warehouses.warehouse_name');

    //     if ($supplierId) {
    //         $inwardsQuery->where('inventory.supplier_id', $supplierId);
    //     }

    //     if ($productId) {
    //         $inwardsQuery->where('inventory.product_id', $productId);
    //     }

    //     if ($warehouseId) {
    //         $inwardsQuery->where('inventory.warehouse_id', $warehouseId);
    //     }

    //     $inwards = $inwardsQuery->get();

    //     // Collect rolls and linked product/variant
    //     $product_items = [];

    //     foreach ($inwards as $inward) {
    //         $rolls = DB::table('rolls')
    //             ->where('rolls.inward_id', $inward->id)
    //             ->leftJoin('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
    //             ->leftJoin('products', 'product_variant.product_id', '=', 'products.id')
    //             ->select(
    //                 'rolls.id',
    //                 'rolls.availability',
    //                 'products.product_name',
    //                 'product_variant.varient_code',
    //                 'product_variant.varient_name'
    //             )
    //             ->get();

    //         if (!$rolls->isEmpty()) {
    //             $product_items[] = [
    //                 'supplier_name' => $inward->supplier_name ?? '-',
    //                 'warehouse' => $inward->warehouse_name ?? '-',
    //                 'rolls' => $rolls
    //             ];
    //         }
    //     }

    //     // Get company details
    //     $company = DB::table('companies')
    //         ->where('company_name', 'Kukreja Interiors')
    //         ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
    //         ->first();

    //     // Generate PDF
    //     $pdf = PDF::loadView('reportstemplate.productsuppliermapping', [
    //         'product_items' => $product_items,
    //         'company' => $company,
    //         'suppliers' => $suppliers,
    //         'products' => $products,
    //         'warehouses' => $warehouses,
    //         'selected_supplier' => $supplierId,
    //         'selected_product' => $productId,
    //         'selected_warehouse' => $request->input('warehouse'),
    //         'is_pdf' => true
    //     ]);

    //     return $pdf->download('product_supplier.pdf');
    // }

    public function productSupplier(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            // Filters
            $supplierId  = $request->filled('supplier') ? (int) $request->input('supplier') : null;
            $productId   = $request->filled('product') ? (int) $request->input('product') : null;
            $variantId   = $request->filled('variant') ? (int) $request->input('variant') : null; // ✅ Added
            $warehouseId = $request->filled('warehouse') ? (int) $request->input('warehouse') : null;

            // Dropdown data
            $suppliers = DB::table('suppliers')
                ->select('id', 'supplier_name')
                ->whereNull('deleted_at')
                ->get();

            $warehouses = DB::table('warehouses')
                ->join('inventory', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->select('warehouses.id', 'warehouses.warehouse_name')
                ->whereNull('warehouses.deleted_at')
                ->groupBy('inventory.warehouse_id')
                ->get();

            $products = collect(); // (loaded via AJAX)

            // Base query for inwards
            $inwardsQuery = DB::table('inventory')
                ->leftJoin('suppliers', 'suppliers.id', '=', 'inventory.supplier_id')
                ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->select('inventory.*', 'suppliers.supplier_name', 'warehouses.warehouse_name');

            if ($supplierId)  $inwardsQuery->where('inventory.supplier_id', $supplierId);
            if ($productId)   $inwardsQuery->where('inventory.product_id', $productId);
            if ($warehouseId) $inwardsQuery->where('inventory.warehouse_id', $warehouseId);

            $inwards = $inwardsQuery->get();

            $product_items = [];

            foreach ($inwards as $inward) {
                // Fetch only available rolls for this inward
                $rollsQuery = DB::table('rolls')
                    ->where('rolls.inward_id', $inward->id)
                    ->where('rolls.availability', '>', 0)
                    ->leftJoin('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
                    ->leftJoin('products', 'products.id', '=', 'product_variant.product_id')
                    ->select(
                        'product_variant.id as variant_id',
                        'products.product_name',
                        'product_variant.varient_code',
                        'product_variant.varient_name',
                        'rolls.availability'
                    );

                // ✅ Apply variant filter
                if ($variantId) {
                    $rollsQuery->where('rolls.varient_id', $variantId);
                }

                $rolls = $rollsQuery->get();

                if ($rolls->isEmpty()) continue;

                $supKey = (int) ($inward->supplier_id ?? 0);

                if (!isset($product_items[$supKey])) {
                    $product_items[$supKey] = [
                        'supplier_id'   => $supKey,
                        'supplier_name' => $inward->supplier_name ?? '-',
                        'rolls'         => []
                    ];
                }

                foreach ($rolls as $r) {
                    $vKey = (int) $r->variant_id;

                    if (!isset($product_items[$supKey]['rolls'][$vKey])) {
                        $product_items[$supKey]['rolls'][$vKey] = (object) [
                            'product_name' => $r->product_name,
                            'varient_code' => $r->varient_code,
                            'varient_name' => $r->varient_name,
                            'availability' => 0,
                        ];
                    }

                    $product_items[$supKey]['rolls'][$vKey]->availability += (float) $r->availability;
                }
            }

            // ✅ Sort products inside each supplier by product_name then varient_name
            $product_items = collect(array_values(array_map(function ($group) {
                $group['rolls'] = collect(array_values($group['rolls']))
                    ->sortBy([
                        ['product_name', 'asc'],
                        ['varient_name', 'asc']
                    ])
                    ->values();
                return $group;
            }, $product_items)));

            // ✅ Sort suppliers by name
            $product_items = $product_items->sortBy('supplier_name')->values();

            // Company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            return view('reportstemplate.productsuppliermapping', [
                'product_items'       => $product_items,
                'company'             => $company,
                'suppliers'           => $suppliers,
                'products'            => $products,
                'warehouses'          => $warehouses,
                'selected_supplier'   => $supplierId,
                'selected_product'    => $productId,
                'selected_variant'    => $variantId, // ✅ Added for form persistence
                'selected_warehouse'  => $request->input('warehouse'),
                'is_pdf'              => false,
            ]);
        }

     public function productSupplierpdf(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            // Filters
            $supplierId  = $request->filled('supplier') ? (int)$request->input('supplier') : null;
            $productId   = $request->filled('product') ? (int)$request->input('product') : null;
            $variantId   = $request->filled('variant') ? (int)$request->input('variant') : null;
            $warehouseId = $request->filled('warehouse') ? (int)$request->input('warehouse') : null;

            // Fetch suppliers (for consistency)
            $suppliers = DB::table('suppliers')
                ->select('id', 'supplier_name')
                ->whereNull('deleted_at')
                ->get();

            $products = collect(); // AJAX loaded in view version

            // Base query for inwards
            $inwardsQuery = DB::table('inventory')
                ->leftJoin('suppliers', 'suppliers.id', '=', 'inventory.supplier_id')
                ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->select('inventory.*', 'suppliers.supplier_name', 'warehouses.warehouse_name');

            if ($supplierId)  $inwardsQuery->where('inventory.supplier_id', $supplierId);
            if ($productId)   $inwardsQuery->where('inventory.product_id', $productId);
            if ($warehouseId) $inwardsQuery->where('inventory.warehouse_id', $warehouseId);

            $inwards = $inwardsQuery->get();

            $product_items = [];

            foreach ($inwards as $inward) {
                // Rolls query with variant filter
                $rollsQuery = DB::table('rolls')
                    ->where('rolls.inward_id', $inward->id)
                    ->where('rolls.availability', '>', 0)
                    ->leftJoin('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
                    ->leftJoin('products', 'products.id', '=', 'product_variant.product_id')
                    ->select(
                        'product_variant.id as variant_id',
                        'products.product_name',
                        'product_variant.varient_code',
                        'product_variant.varient_name',
                        'rolls.availability'
                    );

                if ($variantId) {
                    $rollsQuery->where('rolls.varient_id', $variantId);
                }

                $rolls = $rollsQuery->get();

                if ($rolls->isEmpty()) continue;

                $supKey = (int) ($inward->supplier_id ?? 0);

                if (!isset($product_items[$supKey])) {
                    $product_items[$supKey] = [
                        'supplier_id'   => $supKey,
                        'supplier_name' => $inward->supplier_name ?? '-',
                        'rolls'         => []
                    ];
                }

                foreach ($rolls as $r) {
                    $vKey = (int) $r->variant_id;

                    if (!isset($product_items[$supKey]['rolls'][$vKey])) {
                        $product_items[$supKey]['rolls'][$vKey] = (object) [
                            'product_name' => $r->product_name,
                            'varient_code' => $r->varient_code,
                            'varient_name' => $r->varient_name,
                            'availability' => 0
                        ];
                    }

                    $product_items[$supKey]['rolls'][$vKey]->availability += (float) $r->availability;
                }
            }

            // ✅ Sort products inside each supplier by product_name then varient_name
            $product_items = collect(array_values(array_map(function ($group) {
                $group['rolls'] = collect(array_values($group['rolls']))
                    ->sortBy([
                        ['product_name', 'asc'],
                        ['varient_name', 'asc']
                    ])
                    ->values();
                return $group;
            }, $product_items)));

            // ✅ Sort suppliers by supplier_name
            $product_items = $product_items->sortBy('supplier_name')->values();

            // Company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            // Generate PDF
            $pdf = PDF::loadView('reportstemplate.productsuppliermapping', [
                'product_items'      => $product_items,
                'company'            => $company,
                'suppliers'          => $suppliers,
                'products'           => $products,
                'selected_supplier'  => $supplierId,
                'selected_product'   => $productId,
                'selected_variant'   => $variantId,
                'selected_warehouse' => $warehouseId,
                'is_pdf'             => true
            ]);

            return $pdf->download('product_supplier.pdf');
        }


 public function reservedStocks(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $orderId = $request->input('order_no');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // 🔁 Fallback: If no filters, use today
        $useDefaultDate = !$startDate && !$endDate && !$orderId;
        $defaultDate = date('Y-m-d');

        if ($useDefaultDate) {
            $startDate = $defaultDate;
            $endDate = $defaultDate;
        }

        // 📦 Dropdown for orders
        $orderno = DB::table('orders')
            ->whereNull('deleted_at')
            ->whereIn('status', ['Ordered', 'Measurement'])
            ->select('id', 'order_code')
            ->get();

        // 📊 Query filtered orders
        $raw_orders = DB::table('orders')
            ->whereIn('orders.status', ['Ordered', 'Measurement'])
            ->whereNull('orders.deleted_at')
            ->when($orderId, function ($query) use ($orderId) {
                $query->where('orders.id', $orderId);
            })
            ->when(!$orderId && $startDate, function ($query) use ($startDate) {
                $query->whereDate('orders.order_date', '>=', $startDate);
            })
            ->when(!$orderId && $endDate, function ($query) use ($endDate) {
                $query->whereDate('orders.order_date', '<=', $endDate);
            })
            ->leftJoin('order_roll_mapping', 'order_roll_mapping.order_id', 'orders.id')
            ->leftJoin('customers', 'customers.id', 'orders.customer_id')
            ->leftJoin('order_items', 'order_items.id', 'order_roll_mapping.order_item_id')
            ->leftJoin('product_variant', 'product_variant.id', 'order_items.varient_id')
            ->leftJoin('products', 'product_variant.product_id', 'products.id')
            ->select(
                'orders.id',
                'orders.status',
                'orders.order_code',
                'orders.order_date',
                'customers.customer_name',
                'order_roll_mapping.roll_id',
                'order_roll_mapping.total_quantity',
                'order_roll_mapping.used_quantity',
                'order_roll_mapping.installer_status',
                'product_variant.varient_name',
                'product_variant.varient_code',
                'products.product_name'
            )
            ->orderBy('orders.order_code', 'desc')
            ->get();

        // 🔄 Group by order code
        $order_items = [];
        foreach ($raw_orders->groupBy('order_code') as $order_code => $group) {
            $order_items[] = [
                'order_id' => $order_code,
                'order_items' => $group
            ];
        }

        // 🏢 Company info
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();
        

        return view('reportstemplate.reservedstocks', [
            'order_items' => $order_items,
            'company' => $company,
            'orderno' => $orderno,
            'is_pdf' => false,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'selected_order_no' => $orderId
        ]);
    }


public function reservedStockspdf(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $orderId = $request->input('order_no');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // Default to today's date if no filters are provided
        $useDefaultDate = !$startDate && !$endDate && !$orderId;
        $defaultDate = date('Y-m-d');

                if ($useDefaultDate) {
                    $startDate = $defaultDate;
                    $endDate = $defaultDate;
                }

        // Fetch filtered order data
        $raw_orders = DB::table('orders')
            ->whereIn('orders.status', ['Ordered', 'Measurement'])
            ->whereNull('orders.deleted_at')
            ->when($orderId, function ($query) use ($orderId) {
                $query->where('orders.id', $orderId);
            })
            ->when(!$orderId && $startDate, function ($query) use ($startDate) {
                $query->whereDate('orders.order_date', '>=', $startDate);
            })
            ->when(!$orderId && $endDate, function ($query) use ($endDate) {
                $query->whereDate('orders.order_date', '<=', $endDate);
            })
            ->leftJoin('order_roll_mapping', 'order_roll_mapping.order_id', 'orders.id')
            ->leftJoin('customers', 'customers.id', 'orders.customer_id')
            ->leftJoin('order_items', 'order_items.id', 'order_roll_mapping.order_item_id')
            ->leftJoin('product_variant', 'product_variant.id', 'order_items.varient_id')
            ->leftJoin('products', 'product_variant.product_id', 'products.id')
            ->select(
                'orders.id',
                'orders.status',
                'orders.order_code',
                'orders.order_date',
                'customers.customer_name',
                'order_roll_mapping.roll_id',
                'order_roll_mapping.total_quantity',
                'order_roll_mapping.used_quantity',
                'order_roll_mapping.installer_status',
                'product_variant.varient_name',
                'product_variant.varient_code',
                'products.product_name'
            )
            ->orderBy('orders.order_code', 'desc')
            ->get();

        // Group data by order_code
        $order_items = [];
        foreach ($raw_orders->groupBy('order_code') as $order_code => $group) {
            $order_items[] = [
                'order_id' => $order_code,
                'order_items' => $group
            ];
        }

        // Fetch company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // Generate PDF
        $pdf = PDF::loadView('reportstemplate.reservedstocks', [
            'order_items' => $order_items,
            'company' => $company,
            'is_pdf' => true
        ])->setPaper('a4', 'landscape');

        return $pdf->download('reserved_stocks.pdf');
    }


   public function stockSupply(Request $request)
    {
        // Safely handle large data
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $productId = $request->input('product');

        // Get all distinct products for dropdown
        $products = DB::table('inventory')
            ->leftJoin('products', 'products.id', 'inventory.product_id')
            ->select('inventory.product_id', 'products.id','products.product_name')
            ->groupBy('inventory.product_id', 'products.product_name')
            ->get();

        $product_items = [];

        // Filter product loop
        $productList = $products;
        if (!empty($productId)) {
            $productList = $products->where('product_id', $productId);
        }

        foreach ($productList as $product) {
            $items = DB::table('inventory')
                ->where('inventory.product_id', $product->product_id)
                ->leftJoin('product_variant', 'product_variant.id', 'inventory.product_varient')
                ->leftJoin('pricing', 'product_variant.product_id', 'pricing.product_id')
                ->leftJoin('rolls', 'rolls.inward_id', '=', 'inventory.id')
                ->select(
                    'inventory.quantity',
                    'qunatity_on_hand',
                    'rolls.availability',
                    'pricing.purchase_cost',
                    'pricing.basic_cost',
                    'product_variant.varient_code as varient_code',
                    'product_variant.varient_name as varient_name'
                )
                ->get();

            $product_items[] = [
                'product_name' => $product->product_name,
                'items' => $items
            ];
        }

        // Get company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        return view('reportstemplate.stocksupply', [
            'product_items' => $product_items,
            'company' => $company,
            'products' => $products,
            'selected_product' => $productId,
            'is_pdf' => false
        ]);
    }


    public function stockSupplypdf(Request $request)
    {
        // Safely handle large data
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $productId = $request->input('product');

        // Get all products for filter (used if needed in PDF header or logic)
        $products = DB::table('inventory')
            ->leftJoin('products', 'products.id', 'inventory.product_id')
            ->select('inventory.product_id', 'products.product_name')
            ->groupBy('inventory.product_id', 'products.product_name')
            ->get();

        // Filter list if product is selected
        $productList = $products;
        if (!empty($productId)) {
            $productList = $products->where('product_id', $productId);
        }

        $product_items = [];
        foreach ($productList as $product) {
            $items = DB::table('inventory')
                ->where('inventory.product_id', $product->product_id)
                ->leftJoin('product_variant', 'product_variant.id', 'inventory.product_varient')
                ->leftJoin('pricing', 'product_variant.product_id', 'pricing.product_id')
                ->leftJoin('rolls', 'rolls.inward_id', '=', 'inventory.id')
                ->select(
                    'inventory.quantity',
                    'qunatity_on_hand',
                    'rolls.availability',
                    'pricing.purchase_cost',
                    'pricing.basic_cost',
                    'product_variant.varient_code as varient_code',
                    'product_variant.varient_name as varient_name'
                )
                ->get();

            $product_items[] = [
                'product_name' => $product->product_name,
                'items' => $items
            ];
        }

        // Get company info
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // Generate and return PDF
        $pdf = PDF::loadView('reportstemplate.stocksupply', [
            'product_items' => $product_items,
            'company' => $company,
            'is_pdf' => true
        ])->setPaper('a4', 'portrait');

        return $pdf->download('stock_supply.pdf');
    }


   
   public function supplierStockReport(Request $request)
        {
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            $supplierId = $request->input('supplier');
            $productId = $request->input('product');

            // Fetch all suppliers
            $suppliers = DB::table('suppliers')->select('id', 'supplier_name')->where('deleted_at',null)->get();

            // Fetch products supplied by selected supplier
            $products = collect(); // default empty
            $supplierId = $supplierId ? (int) $supplierId : null;

            if ($supplierId) {
                $products = DB::table('inventory')
                    ->join('products', 'products.id', '=', 'inventory.product_id')
                    ->where('inventory.supplier_id', $supplierId)
                    ->select('products.id', 'products.product_name')
                    ->groupBy('products.id', 'products.product_name')
                    ->where('products.deleted_at',null)
                    ->get();
            }


            // Fetch warehouses based on selected product
                $warehouses = collect(); // default empty
                if ($productId) {
                    $warehouses = DB::table('inventory')
                        ->join('rolls', 'inventory.id', '=', 'rolls.inward_id')
                        ->join('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
                        ->join('products', 'products.id', '=', 'product_variant.product_id')
                      
                        ->join('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                        ->where('products.id', $productId)
                        ->when($supplierId, function ($query) use ($supplierId) {
                            return $query->where('inventory.supplier_id', $supplierId);
                        })
                        ->select('warehouses.id', 'warehouses.warehouse_name')
                        ->groupBy('warehouses.id', 'warehouses.warehouse_name')
                        ->get();
                }



            // Get inward entries filtered by supplier
            $inwardsQuery = DB::table('inventory')
                ->leftJoin('suppliers', 'suppliers.id', 'inventory.supplier_id')
                ->leftJoin('warehouses', 'warehouses.id', 'inventory.warehouse_id')
                ->leftJoin('purchase_orders', 'purchase_orders.po_number', 'inventory.po_number')
                    ->select(
                        'inventory.id as inward_id',
                         'inventory.inward_no',
                        'suppliers.supplier_name',
                        'inventory.supplier_id',
                        'warehouses.warehouse_name',
                        'purchase_orders.grand_total'
                    );


            if ($supplierId) {
                $inwardsQuery->where('inventory.supplier_id', $supplierId);
            }

            $warehouseId = $request->input('warehouse');
                if ($warehouseId) {
                    $inwardsQuery->where('inventory.warehouse_id', $warehouseId);
                }


            $inwards = $inwardsQuery->get();
            //dd( $inwards);

            $inward_items = [];

            foreach ($inwards as $inward) {
                $rollsQuery = DB::table('rolls')
                    ->where('rolls.inward_id', $inward->inward_id)
                    ->where('rolls.availability', '>', 0) 
                    ->leftJoin('product_variant', 'product_variant.id', 'rolls.varient_id')
                    ->leftJoin('products', 'product_variant.product_id', 'products.id')
                    ->leftJoin('pricing', 'product_variant.product_id', 'pricing.product_id')
                    ->select(
                        'rolls.*',
                        'products.id as product_id',
                        'products.product_name',
                        'pricing.basic_cost',
                        'product_variant.varient_name',
                        'product_variant.varient_code'
                    );

                if ($productId) {
                    $rollsQuery->where('products.id', $productId);
                }

                $rolls = $rollsQuery->orderByRaw('CAST(rolls.availability AS DECIMAL(10,2)) ASC')->get();

                if (!$rolls->isEmpty()) {
                    $inward_items[] = [
                        'inward_id' => $inward->inward_no,
                        'supplier_name' => $inward->supplier_name,
                        'warehouse' => $inward->warehouse_name,
                        'grand_total' => $inward->grand_total,
                        'items' => $rolls
                    ];
                }
            }

            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            //    dd($inward_items);

            return view('reportstemplate.supplierstockreport', [
                'inward_items' => $inward_items,
                'suppliers' => $suppliers,
                'products' => $products,
                'warehouses' => $warehouses,
                'selected_supplier' => $supplierId,
                'selected_product' => $productId,
                 'selected_warehouse' => $warehouseId,
                'company' => $company,
                'is_pdf' => false
            ]);
        }



      public function supplierStockReportpdf(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $supplierId = $request->input('supplier');
        $productId = $request->input('product');
        $warehouseId = $request->input('warehouse');

        // Get inwards filtered by supplier and warehouse
        $inwardsQuery = DB::table('inventory')
            ->leftJoin('suppliers', 'suppliers.id', 'inventory.supplier_id')
            ->leftJoin('warehouses', 'warehouses.id', 'inventory.warehouse_id')
            ->leftJoin('purchase_orders', 'purchase_orders.po_number', 'inventory.po_number')
            ->select(
                'inventory.id as inward_id',
                'inventory.inward_no',
                'suppliers.supplier_name',
                'inventory.supplier_id',
                'warehouses.warehouse_name',
                'purchase_orders.grand_total'
            );

        if ($supplierId) {
            $inwardsQuery->where('inventory.supplier_id', $supplierId);
        }

        if ($warehouseId) {
            $inwardsQuery->where('inventory.warehouse_id', $warehouseId);
        }

        $inwards = $inwardsQuery->get();

        $inward_items = [];

        foreach ($inwards as $inward) {
            $rollsQuery = DB::table('rolls')
                ->where('rolls.inward_id', $inward->inward_id)
                ->where('rolls.availability', '>', 0) // Exclude zero availability
                ->leftJoin('product_variant', 'product_variant.id', 'rolls.varient_id')
                ->leftJoin('products', 'product_variant.product_id', 'products.id')
                ->leftJoin('pricing', 'product_variant.product_id', 'pricing.product_id')
                ->select(
                    'rolls.*',
                    'products.id as product_id',
                    'products.product_name',
                    'pricing.basic_cost',
                    'product_variant.varient_name',
                    'product_variant.varient_code'
                );

            if ($productId) {
                $rollsQuery->where('products.id', $productId);
            }

            // Sort rolls by availability (lowest first)
            $rolls = $rollsQuery->orderByRaw('CAST(rolls.availability AS DECIMAL(10,2)) ASC')->get();

            if (!$rolls->isEmpty()) {
                $inward_items[] = [
                    'inward_id' => $inward->inward_no,
                    'supplier_name' => $inward->supplier_name,
                    'warehouse' => $inward->warehouse_name,
                    'grand_total' => $inward->grand_total,
                    'items' => $rolls
                ];
            }
        }

        // Company info
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        $pdf = PDF::loadView('reportstemplate.supplierstockreport', [
            'inward_items' => $inward_items,
            'company' => $company,
            'is_pdf' => true
        ]);

        return $pdf->download('supplier_stock_report.pdf');
    }




     public function todaysCutting(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $startDateTime = $startDate . ' 00:00:00';
        $endDateTime = $endDate . ' 23:59:59';

        // Step 1: Fetch only orders that have at least one item with Cutting Done
        $orders = DB::table('orders')
            ->leftJoin('customers', 'customers.id', 'orders.customer_id')
            ->select('orders.id', 'customers.customer_name', 'orders.order_code','orders.customer_discount_amount','orders.order_date')
            ->whereBetween('orders.updated_at', [$startDateTime, $endDateTime])
            ->whereExists(function ($query) use ($startDateTime, $endDateTime) {
                $query->select(DB::raw(1))
                    ->from('order_items')
                    ->join('order_roll_mapping', 'order_roll_mapping.order_item_id', 'order_items.id')
                    ->whereColumn('order_items.order_id', 'orders.id')
                    ->where('order_roll_mapping.installer_status', 'Cutting Done')
                    ->whereBetween('order_items.updated_at', [$startDateTime, $endDateTime]);
            })
            ->get();

        // Step 2: Loop and fetch order_items with Cutting Done
        $order_receipts = [];
        foreach ($orders as $order) {
            $order_items = DB::table('order_items')
                ->leftJoin('order_roll_mapping', 'order_roll_mapping.order_item_id', 'order_items.id')
                ->leftJoin('orders', 'order_items.order_id', '=', 'orders.id')
                ->leftJoin('product_variant', 'order_items.varient_id', '=', 'product_variant.id')
                ->leftJoin('products', 'products.id', '=', 'product_variant.product_id')
                ->where('order_items.order_id', $order->id)
                ->where('order_roll_mapping.installer_status', 'Cutting Done')
                ->whereBetween('order_items.updated_at', [$startDateTime, $endDateTime])
                  ->select(
                        'order_items.*', // ensures all original fields like price are fetched
                        'product_variant.varient_code',
                         'order_roll_mapping.roll_id',
                         'order_roll_mapping.installer_status',
                        'order_roll_mapping.roll_quantity',
                        'order_roll_mapping.used_quantity',
                         'order_roll_mapping.available_quantity',
                        'product_variant.varient_name',
                        'products.product_name'
                    )
                ->get();

            if ($order_items->isNotEmpty()) {
                $order_receipts[] = (object) [
                    'id' => $order->order_code,
                    'customer_name' => $order->customer_name,
                    'order_items' => $order_items,
                      'customer_discount_amount' => $order->customer_discount_amount,
                      'order_date'=>$order->order_date
                ];
            }
        }

        // Company info
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

           // dd($order_receipts);

        return view('reportstemplate.todayscutting', [
            'order_receipts' => $order_receipts,
            'company' => $company,
            'is_pdf' => false
        ]);
    }


     public function todaysCuttingpdf(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');
        $startDateTime = $startDate . ' 00:00:00';
        $endDateTime = $endDate . ' 23:59:59';

        // Step 1: Fetch only orders that have at least one item with Cutting Done
        $orders = DB::table('orders')
            ->leftJoin('customers', 'customers.id', 'orders.customer_id')
            ->select(
                'orders.id',
                'customers.customer_name',
                'orders.order_code',
                'orders.customer_discount_amount',
                'orders.order_date'
            )
            ->whereBetween('orders.updated_at', [$startDateTime, $endDateTime])
            ->whereExists(function ($query) use ($startDateTime, $endDateTime) {
                $query->select(DB::raw(1))
                    ->from('order_items')
                    ->join('order_roll_mapping', 'order_roll_mapping.order_item_id', 'order_items.id')
                    ->whereColumn('order_items.order_id', 'orders.id')
                    ->where('order_roll_mapping.installer_status', 'Cutting Done')
                    ->whereBetween('order_items.updated_at', [$startDateTime, $endDateTime]);
            })
            ->get();

        // Step 2: Loop and fetch order_items with Cutting Done
        $order_receipts = [];
        foreach ($orders as $order) {
            $order_items = DB::table('order_items')
                ->leftJoin('order_roll_mapping', 'order_roll_mapping.order_item_id', 'order_items.id')
                ->leftJoin('orders', 'order_items.order_id', '=', 'orders.id')
                ->leftJoin('product_variant', 'order_items.varient_id', '=', 'product_variant.id')
                ->leftJoin('products', 'products.id', '=', 'product_variant.product_id')
                ->where('order_items.order_id', $order->id)
                ->where('order_roll_mapping.installer_status', 'Cutting Done')
                ->whereBetween('order_items.updated_at', [$startDateTime, $endDateTime])
                ->select(
                    'order_items.*',
                    'product_variant.varient_code',
                    'product_variant.varient_name',
                    'products.product_name',
                    'order_roll_mapping.roll_id',
                    'order_roll_mapping.installer_status',
                    'order_roll_mapping.roll_quantity',
                    'order_roll_mapping.used_quantity',
                    'order_roll_mapping.available_quantity'
                )
                ->get();

            if ($order_items->isNotEmpty()) {
                $order_receipts[] = (object)[
                    'id' => $order->order_code,
                    'customer_name' => $order->customer_name,
                    'order_items' => $order_items,
                    'customer_discount_amount' => $order->customer_discount_amount,
                    'order_date' => $order->order_date,
                ];
            }
        }

        // Company Info
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // Generate PDF
        $pdf = PDF::loadView('reportstemplate.todayscutting', [
            'order_receipts' => $order_receipts,
            'company' => $company,
            'is_pdf' => true
        ]);

        return $pdf->download('todays_cutting.pdf');
    }

      public function todaysCash(Request $request)
        {

        

            // Force today's date if not set in URL
            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');

            // If no date input was submitted (user just visited the page)
            if (!$request->has('start_date') && !$request->has('end_date')) {
                $startDate = date('Y-m-d');
                $endDate = date('Y-m-d');
            }
            // dd($request->all(), $startDate, $endDate);
            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';

                $startDate = $request->input('start_date') ?: date('Y-m-d');
                        $endDate = $request->input('end_date') ?: date('Y-m-d');

                        $query = DB::table('order_receipt')
                            ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
                            ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
                            ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                            ->select(
                                'order_receipt.*',
                                'orders.order_code',
                                'order_transaction.payment_mode',
                                'order_transaction.pay_amount',
                                'order_transaction.cheque_number',
                                'order_transaction.transaction_date',
                                'customers.customer_name',
                                'banks.bank_name',
                                'order_receipt.receipt_id',
                                'order_transaction.remark',
                                'order_receipt.receipt_type'
                            )
                            ->where('order_transaction.payment_mode', 'Cash');

                        // Apply date filter based on request
                        if ($request->has(['start_date', 'end_date'])) {
                            $query->whereBetween('order_transaction.transaction_date', [
                                $startDate . ' 00:00:00',
                                $endDate . ' 23:59:59',
                            ]);
                        } else {
                            // Show today's data by default
                            $query->whereDate('order_transaction.transaction_date', $startDate);
                        }

                        $order_receipts = $query
                            ->orderBy('order_transaction.transaction_date', 'desc')
                            ->get();

            $grouped_data = [];
            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode;
                $customer = $receipt->customer_name;

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer])) {
                    $grouped_data[$payment_method][$customer] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer]['orders'][] = [
                    'order_id' => $receipt->order_code,
                    'amount' => number_format($receipt->pay_amount, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'remark' => $receipt->remark,
                    'cheq_no' => $receipt->cheque_number,
                    'rcpt_type' => $receipt->receipt_type,
                    'transaction_date' => $receipt->transaction_date,
                ];

                $grouped_data[$payment_method][$customer]['total_amount'] += $receipt->pay_amount;
            }

            // Company info
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            return view('reportstemplate.todayscash', [
                'grouped_data' => $grouped_data,
                'company' => $company,
                'is_pdf' => false,
                'start_date' => $startDate,
                'end_date' => $endDate
            ]);
        }



        public function todaysCashpdf(Request $request)
        {
            // Safely handle large data
            ini_set('memory_limit', '1024M');
            set_time_limit(300);

            // Handle start and end date with default to today
            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');

            if (!$request->has('start_date') && !$request->has('end_date')) {
                $startDate = date('Y-m-d');
                $endDate = date('Y-m-d');
            }

            $query = DB::table('order_receipt')
                ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
                ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
                ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                ->select(
                    'order_receipt.*',
                    'orders.order_code',
                    'order_transaction.payment_mode',
                    'order_transaction.pay_amount',
                    'order_transaction.cheque_number',
                    'order_transaction.transaction_date',
                    'customers.customer_name',
                    'banks.bank_name',
                     'order_receipt.receipt_id',
                    'order_transaction.remark',
                )
                ->where('order_transaction.payment_mode', 'Cash');

            // Apply filtering
            if ($request->has(['start_date', 'end_date'])) {
                $query->whereBetween('order_transaction.transaction_date', [
                    $startDate . ' 00:00:00',
                    $endDate . ' 23:59:59',
                ]);
            } else {
                $query->whereDate('order_transaction.transaction_date', $startDate);
            }

            $order_receipts = $query->orderBy('order_transaction.transaction_date', 'desc')->get();

            // Group the data
            $grouped_data = [];

            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode;
                $customer = $receipt->customer_name;

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer])) {
                    $grouped_data[$payment_method][$customer] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer]['orders'][] = [
                    'order_id' => $receipt->order_code,
                    'amount' => number_format($receipt->pay_amount, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'remark' => $receipt->remark,
                    'cheq_no' => $receipt->cheque_number,
                    'rcpt_type' => $receipt->receipt_type,
                    'transaction_date' => $receipt->transaction_date,
                ];

                $grouped_data[$payment_method][$customer]['total_amount'] += $receipt->pay_amount;
            }

            // Get company info
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

            // Generate PDF
            $pdf = PDF::loadView('reportstemplate.todayscash', [
                'grouped_data' => $grouped_data,
                'company' => $company,
                'is_pdf' => true,
                'start_date' => $startDate,
                'end_date' => $endDate
            ]);

            return $pdf->download('todays_cash.pdf');
        }

    public function todaysCollection(Request $request){

            // Safely handle large data
            ini_set('memory_limit', '1024M'); // Increase memory limit
            set_time_limit(300);              // Increase max execution time to 5 minutes
            // If no date input was submitted (user just visited the page)
            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');
            $payment_mode = $request->input('payment_mode');
            $bank = $request->input('bank');
             $order = $request->input('order');
             $customerId = $request->input('customer');


            if (!$request->has('start_date') && !$request->has('end_date')) {
                $startDate = date('Y-m-d');
                $endDate = date('Y-m-d');
            }
            // dd($request->all(), $startDate, $endDate);
            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';

             $startDate = $request->input('start_date') ?: date('Y-m-d');
             $endDate = $request->input('end_date') ?: date('Y-m-d');

            $query = DB::table('order_receipt')
            ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
            ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
            ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
            ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                ->select(
                    'order_receipt.*',
                    'orders.order_code',
                    'order_transaction.bank_name',
                    'order_transaction.ifsc_code',
                    'order_transaction.account_number',
                    'order_transaction.reference_id',
                     'order_transaction.transaction_date',
                    'order_transaction.pay_amount as amount_paid',
                    'order_transaction.cheque_number',
                    'order_transaction.remark as transaction_remark',
                     'order_transaction.payment_mode',

                    'customers.customer_name',
                    'banks.bank_name'
                );
               
                 if ($request->has(['start_date', 'end_date'])) {
                            $query->whereBetween('order_transaction.transaction_date', [
                                $startDate . ' 00:00:00',
                                $endDate . ' 23:59:59',
                            ]);
                        } else {
                            // Show today's data by default
                            $query->whereDate('order_transaction.transaction_date', $startDate);
                        }

            if (!empty($payment_mode)) {
                        $query->where('order_transaction.payment_mode', $payment_mode);
                    }

                    
            if (!empty($bank)) {
                $query->where('order_transaction.bank_name', $bank);
            }

                if (!empty($order)) {
                        $query->where('order_transaction.order_id', $order);
                    }

                if (!empty($customerId)) {
                    $query->where('order_receipt.customer_name', $customerId);
                }

                $order_receipts = $query  
                ->orderBy('order_transaction.payment_mode') // Order by Payment Method first
                ->orderBy('customers.customer_name') // Then order by Customer Name
                ->orderBy('order_transaction.id') // Order by Order No
                
                ->get();
            // dd($order_receipts);
            $grouped_data = [];

            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode;
                $customer = $receipt->customer_name;

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer])) {
                    $grouped_data[$payment_method][$customer] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer]['orders'][] = [
                    'order_id' => $receipt->order_code,
                    'amount' => number_format($receipt->amount_paid, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'ifsc_code' => $receipt->ifsc_code,
                    'account_number' => $receipt->account_number,
                    'remark' => $receipt->transaction_remark,
                    'cheq_no' => $receipt->cheque_number,
                     'reference_id' => $receipt->reference_id,
                     'transaction_date' => $receipt->transaction_date,
                    'rcpt_type' => $receipt->receipt_type
                ];

                $grouped_data[$payment_method][$customer]['total_amount'] += $receipt->amount_paid;
            }

            //  dd($grouped_data);

            // Fetch Kukreja Interiors company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
                ->first();
            $banks = DB::table('banks')->whereNull('deleted_at')->get();
             $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();
            $customers = DB::table('customers')->whereNull('deleted_at')->get();

            return view('reportstemplate.todayscollections', [
                'grouped_data' => $grouped_data,
                'company'=>$company,
                'banks'=>$banks,
                 'orders'=>$orders,
                'customers'=>$customers,
                'is_pdf' => false 
                ]);
    }


        public function todaysCollectionpdf(Request $request)
        {
            // Safely handle large data
            ini_set('memory_limit', '1024M'); // Increase memory limit
            set_time_limit(300);              // Increase max execution time to 5 minutes

            $startDate = $request->input('start_date') ?? date('Y-m-d');
            $endDate = $request->input('end_date') ?? date('Y-m-d');
             $payment_mode = $request->input('payment_mode');
            $bank = $request->input('bank');
             $order = $request->input('order');
            $customerId = $request->input('customer');

            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';

            $query = DB::table('order_receipt')
                ->leftJoin('order_transaction', 'order_transaction.order_receipt_id', '=', 'order_receipt.id')
                ->leftJoin('orders', 'order_receipt.order_id', '=', 'orders.id')
                ->leftJoin('banks', 'banks.id', '=', 'order_transaction.bank_name')
                ->leftJoin('customers', 'customers.id', '=', 'order_receipt.customer_name')
                ->select(
                    'order_receipt.*',
                    'orders.order_code',
                    'order_transaction.bank_name',
                    'order_transaction.ifsc_code',
                    'order_transaction.account_number',
                    'order_transaction.reference_id',
                    'order_transaction.transaction_date',
                    'order_transaction.pay_amount as amount_paid',
                    'order_transaction.cheque_number',
                    'order_transaction.remark as transaction_remark',
                    'order_transaction.payment_mode',
                    'customers.customer_name',
                    'banks.bank_name'
                );

            // Apply date filter on transaction_date
            if ($request->has(['start_date', 'end_date'])) {
                $query->whereBetween('order_transaction.transaction_date', [$startDateTime, $endDateTime]);
            } else {
                $query->whereDate('order_transaction.transaction_date', $startDate);
            }

            if (!empty($payment_mode)) {
                        $query->where('order_transaction.payment_mode', $payment_mode);
                    }

                    
            if (!empty($bank)) {
                $query->where('order_transaction.bank_name', $bank);
            }

            
                    if (!empty($order)) {
                    $query->where('order_transaction.order_id', $order);
                   }

                    

                     if (!empty($customerId)) {
                            $query->where('order_receipt.customer_name', $customerId);
                        }

            $order_receipts = $query
                ->orderBy('order_transaction.payment_mode')
                ->orderBy('customers.customer_name')
                ->orderBy('order_transaction.id')
                ->get();

            // Group data by payment_mode and customer
            $grouped_data = [];

            foreach ($order_receipts as $receipt) {
                $payment_method = $receipt->payment_mode;
                $customer = $receipt->customer_name;

                if (!isset($grouped_data[$payment_method])) {
                    $grouped_data[$payment_method] = [];
                }

                if (!isset($grouped_data[$payment_method][$customer])) {
                    $grouped_data[$payment_method][$customer] = [
                        'total_amount' => 0,
                        'orders' => []
                    ];
                }

                $grouped_data[$payment_method][$customer]['orders'][] = [
                    'order_id' => $receipt->order_code,
                    'amount' => number_format($receipt->amount_paid, 2),
                    'receipt_id' => $receipt->receipt_id,
                    'bank' => $receipt->bank_name,
                    'ifsc_code' => $receipt->ifsc_code,
                    'account_number' => $receipt->account_number,
                    'remark' => $receipt->transaction_remark,
                    'cheq_no' => $receipt->cheque_number,
                    'reference_id' => $receipt->reference_id,
                    'transaction_date' => $receipt->transaction_date,
                    'rcpt_type' => $receipt->receipt_type
                ];

                $grouped_data[$payment_method][$customer]['total_amount'] += $receipt->amount_paid;
            }

            // Fetch Kukreja Interiors company details
            $company = DB::table('companies')
                ->where('company_name', 'Kukreja Interiors')
                ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
                ->first();

                    $orders = DB::table('orders')->where('payment_status','<>','Pending')->select('id','order_code')->whereNull('deleted_at')->get();
            $customers = DB::table('customers')->whereNull('deleted_at')->get();
              $banks = DB::table('banks')->whereNull('deleted_at')->get();
            // Generate PDF using the same Blade template
            $pdf = PDF::loadView('reportstemplate.todayscollections', [
                'grouped_data' => $grouped_data,
                'company' => $company,
                'banks'=>$banks,
                 'orders'=>$orders,
                'customers'=>$customers,
                'is_pdf' => true
            ]);

            return $pdf->download('todays_collection.pdf');
        }

    // public function todaysInward(){

    //         // Safely handle large data
    //         ini_set('memory_limit', '1024M'); // Increase memory limit
    //         set_time_limit(300);              // Increase max execution time to 5 minutes

    //         $inward_company = DB::table('inventory')
    //         ->whereDate('inward_date', '=', date('Y-m-d'))
    //         ->leftJoin('companies', 'companies.id', '=', 'inventory.company_id')
    //         ->select('inventory.company_id', 'companies.company_name')
    //         ->groupBy('company_id', 'company_name')
    //         ->get();


    //         $inward_data = [];
            
    //         foreach ($inward_company as $company) {
    //             $inward_items = DB::table('inventory')
    //                 ->whereDate('inward_date', '=', date('Y-m-d'))
    //                 ->where('inventory.company_id', $company->company_id)
    //                 ->leftJoin('purchase_orders', 'purchase_orders.id', '=', 'inventory.po_number')
    //                 ->leftJoin('estimated_po', 'estimated_po.inward_id', '=', 'inventory.id')
    //                 ->leftJoin('products', 'products.id', '=', 'inventory.product_id')
    //                 ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
    //                 ->select(
    //                     'inventory.*',
    //                     'inventory.supplier_id', 
    //                     'inventory.inward_no',
    //                     'products.product_name',
    //                     'purchase_orders.id as purchase_order_id',
    //                     'warehouses.warehouse_name'
    //                 )
    //                 ->get();
                    
    //             // Grouping by company
    //             if (!isset($inward_data[$company->company_name])) {
    //                 $inward_data[$company->company_name] = [];
    //             }
            
    //             foreach ($inward_items as $item) {
    //                 $supplier_id = $item->inward_no;
    //                 $purchase_order_id = $item->purchase_order_id;
            
    //                 // Initialize supplier group if not set
    //                 if (!isset($inward_data[$company->company_name][$supplier_id])) {
    //                     $inward_data[$company->company_name][$supplier_id] = [
    //                         'supplier_id' => $supplier_id,
    //                         'warehouse' => $item->warehouse_name,
    //                         'purchase_orders' => []
    //                     ];
    //                 }
            
    //                 // Initialize purchase order group inside supplier
    //                 if (!isset($inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id])) {
    //                     $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id] = [
    //                         'purchase_order_id' => $purchase_order_id,
    //                         'items' => []
    //                     ];
    //                 }
            
    //                 // Fetch related purchase order items for the current inward item
    //                 $purchase_order_items = DB::table('estimated_po')
    //                     ->leftjoin('purchase_order_items','estimated_po.po_detail_id','=','purchase_order_items.id')
    //                     ->leftJoin('product_variant', 'product_variant.id', '=', 'estimated_po.varient_id')
    //                     ->leftJoin('units', 'units.id', '=', 'estimated_po.unit')
    //                     ->where('estimated_po.po_id', $purchase_order_id)
    //                     ->where('estimated_po.inward_id', $item->id)  
    //                     ->select(
    //                         'estimated_po.quantity',
    //                         'estimated_po.no_of_rolls',
    //                         'units.unit_name',
    //                         'product_variant.varient_name',
    //                         'purchase_order_items.gst_percent',
    //                         'estimated_po.rate',
    //                         'estimated_po.amount as total',
    //                         'estimated_po.no_of_rolls'
    //                     )
    //                     ->get();
    //                   //  dd($purchase_order_items);
                    
    //                 // Add item inside the purchase order group
    //                 $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id]['items'][] = [
    //                     'id' => $item->id,
    //                     'invoice_id' => $item->inward_no,
    //                     'product_name' => $item->product_name,
    //                     'purchase_order_items' => $purchase_order_items, // Nested array of items
    //                 ];
    //             }
    //         }
            
    //          //dd($inward_data);
    //         // Fetch Kukreja Interiors company details
    //         $company = DB::table('companies')
    //             ->where('company_name', 'Kukreja Interiors')
    //             ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
    //             ->first();

    //         return view('reportstemplate.todaysInward', [
    //             'inward_data' => $inward_data,
    //             'company'=>$company,
    //             'is_pdf' => false 
    //             ]);
    // }

   

     public function todaysInward(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');

        // Fetch inward companies
        $inward_company = DB::table('inventory')
            ->whereBetween('inward_date', [$startDate, $endDate])
            ->leftJoin('companies', 'companies.id', '=', 'inventory.company_id')
            ->select('inventory.company_id', 'companies.company_name')
            ->groupBy('company_id', 'company_name')
            ->get();

        $inward_data = [];

        foreach ($inward_company as $company) {
            $inward_items = DB::table('inventory')
                ->whereBetween('inward_date', [$startDate, $endDate])
                ->where('inventory.company_id', $company->company_id)
                ->leftJoin('purchase_orders', 'purchase_orders.id', '=', 'inventory.po_number')
                ->leftJoin('estimated_po', 'estimated_po.inward_id', '=', 'inventory.id')
                ->leftJoin('products', 'products.id', '=', 'inventory.product_id')
                ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->select(
                    'inventory.*',
                    'inventory.supplier_invoice_no',
                    'inventory.inward_no',
                    'products.product_name',
                    'purchase_orders.id as purchase_order_id',
                    'warehouses.warehouse_name'
                )
                ->get();

            if (!isset($inward_data[$company->company_name])) {
                $inward_data[$company->company_name] = [];
            }

            foreach ($inward_items as $item) {
                $supplier_id = $item->supplier_invoice_no;
                $purchase_order_id = $item->purchase_order_id;

                if (!isset($inward_data[$company->company_name][$supplier_id])) {
                    $inward_data[$company->company_name][$supplier_id] = [
                        'supplier_id' => $supplier_id,
                        'warehouse' => $item->warehouse_name,
                        'inward_date' => $item->inward_date,
                        'purchase_orders' => []
                    ];
                }

                if (!isset($inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id])) {
                    $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id] = [
                        'purchase_order_id' => $purchase_order_id,
                        'items' => []
                    ];
                }

                $purchase_order_items = DB::table('estimated_po')
                    ->leftJoin('purchase_order_items', 'estimated_po.po_detail_id', '=', 'purchase_order_items.id')
                    ->leftJoin('product_variant', 'product_variant.id', '=', 'estimated_po.varient_id')
                    ->leftJoin('units', 'units.id', '=', 'estimated_po.unit')
                    ->where('estimated_po.po_id', $purchase_order_id)
                    ->where('estimated_po.inward_id', $item->id)
                    ->select(
                        'estimated_po.id as est_id',
                        'estimated_po.quantity',
                        'estimated_po.no_of_rolls',
                        'units.unit_name',
                        'product_variant.varient_name',
                        'product_variant.varient_code',
                        'purchase_order_items.gst_percent',
                        'estimated_po.rate',
                        'estimated_po.amount as total',
                        'estimated_po.no_of_rolls'
                    )
                    ->get();

                foreach ($purchase_order_items as $po_item) {
                    // Fetch rolls for each estimated_po record
                    $rolls = DB::table('rolls')
                        ->where('inward_id', $item->id)
                       // ->where('estimated_po_id', $po_item->est_id)
                        ->select('id as roll_no')
                        ->get();

                    $po_item->rolls = $rolls;
                }
               // dd($purchase_order_items);

                $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id]['items'][] = [
                    'id' => $item->id,
                    'invoice_id' => $item->inward_no,
                    'product_name' => $item->product_name,
                    'purchase_order_items' => $purchase_order_items,
                ];

               // dd($inward_data);
            }
        }

        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        return view('reportstemplate.todaysInward', [
            'inward_data' => $inward_data,
            'company' => $company,
            'is_pdf' => false
        ]);
    }

    public function todaysInwardpdf(Request $request)
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(300);

        $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');

        // Fetch inward companies
        $inward_company = DB::table('inventory')
            ->whereBetween('inward_date', [$startDate, $endDate])
            ->leftJoin('companies', 'companies.id', '=', 'inventory.company_id')
            ->select('inventory.company_id', 'companies.company_name')
            ->groupBy('company_id', 'company_name')
            ->get();

        $inward_data = [];

        foreach ($inward_company as $company) {
            $inward_items = DB::table('inventory')
                ->whereBetween('inward_date', [$startDate, $endDate])
                ->where('inventory.company_id', $company->company_id)
                ->leftJoin('purchase_orders', 'purchase_orders.id', '=', 'inventory.po_number')
                ->leftJoin('estimated_po', 'estimated_po.inward_id', '=', 'inventory.id')
                ->leftJoin('products', 'products.id', '=', 'inventory.product_id')
                ->leftJoin('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->select(
                    'inventory.*',
                    'inventory.supplier_id',
                    'inventory.supplier_invoice_no',
                    'inventory.inward_no',
                    'products.product_name',
                    'purchase_orders.id as purchase_order_id',
                    'warehouses.warehouse_name'
                )
                ->get();

            if (!isset($inward_data[$company->company_name])) {
                $inward_data[$company->company_name] = [];
            }

            foreach ($inward_items as $item) {
                $supplier_id = $item->supplier_invoice_no;
                $purchase_order_id = $item->purchase_order_id;

                if (!isset($inward_data[$company->company_name][$supplier_id])) {
                    $inward_data[$company->company_name][$supplier_id] = [
                        'supplier_id' => $supplier_id,
                        'warehouse' => $item->warehouse_name,
                        'inward_date' => $item->inward_date,
                        'purchase_orders' => []
                    ];
                }

                if (!isset($inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id])) {
                    $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id] = [
                        'purchase_order_id' => $purchase_order_id,
                        'items' => []
                    ];
                }

                // Fetch purchase order items for this inward
                $purchase_order_items = DB::table('estimated_po')
                    ->leftJoin('purchase_order_items', 'estimated_po.po_detail_id', '=', 'purchase_order_items.id')
                    ->leftJoin('product_variant', 'product_variant.id', '=', 'estimated_po.varient_id')
                    ->leftJoin('units', 'units.id', '=', 'estimated_po.unit')
                    ->where('estimated_po.po_id', $purchase_order_id)
                    ->where('estimated_po.inward_id', $item->id)
                    ->select(
                        'estimated_po.id as est_id',
                        'estimated_po.quantity',
                        'estimated_po.no_of_rolls',
                        'units.unit_name',
                        'product_variant.varient_name',
                        'product_variant.varient_code',
                        'purchase_order_items.gst_percent',
                        'estimated_po.rate',
                        'estimated_po.amount as total',
                        'estimated_po.no_of_rolls'
                    )
                    ->get();

                // Attach rolls to each estimated_po item
                foreach ($purchase_order_items as $po_item) {
                    $rolls = DB::table('rolls')
                        ->where('inward_id', $item->id)
                        // ->where('estimated_po_id', $po_item->est_id) // Uncomment if roll mapping exists
                        ->select('id as roll_no')
                        ->get();

                    $po_item->rolls = $rolls;
                }

                $inward_data[$company->company_name][$supplier_id]['purchase_orders'][$purchase_order_id]['items'][] = [
                    'id' => $item->id,
                    'invoice_id' => $item->inward_no,
                    'product_name' => $item->product_name,
                    'purchase_order_items' => $purchase_order_items,
                ];
            }
        }

        // Fetch company details for header/footer
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state', 'phone1', 'gst_no')
            ->first();

        // Generate PDF
        $pdf = PDF::loadView('reportstemplate.todaysInward', [
            'inward_data' => $inward_data,
            'company' => $company,
            'is_pdf' => true
        ]);

        return $pdf->download('todays_inward.pdf');
    }

    public function todaysOrder(Request $request){

        // Safely handle large data
        ini_set('memory_limit', '1024M'); // Increase memory limit
        set_time_limit(300);              // Increase max execution time to 5 minutes

         $startDate = $request->input('start_date') ?? date('Y-m-d');
        $endDate = $request->input('end_date') ?? date('Y-m-d');

         $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';


        $orders = DB::table('orders')
      //  ->where('order_date', date('Y-m-d'))
        ->whereBetween('order_date', [$startDateTime, $endDateTime])
        ->leftJoin('customers', 'customers.id', 'orders.customer_id')
        ->select('orders.*', 'customers.customer_name')
        ->get();
        // return view('reportstemplate.todaysOrder', compact('orders'));

         // Fetch Kukreja Interiors company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
            ->first();

        return view('reportstemplate.todaysOrder', [
            'orders' => $orders,
            'company'=>$company,
            'is_pdf' => false 
            ]);
    }

     public function todaysOrderpdf(Request $request){

        // Safely handle large data
        ini_set('memory_limit', '1024M'); // Increase memory limit
        set_time_limit(300);              // Increase max execution time to 5 minutes
          $startDate = $request->input('start_date') ?? date('Y-m-d');
          $endDate = $request->input('end_date') ?? date('Y-m-d');
            $startDateTime = $startDate . ' 00:00:00';
            $endDateTime = $endDate . ' 23:59:59';
        

        $orders = DB::table('orders')
        // ->where('order_date', date('Y-m-d'))
        ->whereBetween('order_date', [$startDateTime, $endDateTime])
        ->leftJoin('customers', 'customers.id', 'orders.customer_id')
        ->select('orders.*', 'customers.customer_name')
        ->get();
        // return view('reportstemplate.todaysOrder', compact('orders'));

         // Fetch Kukreja Interiors company details
        $company = DB::table('companies')
            ->where('company_name', 'Kukreja Interiors')
            ->select('company_name', 'address', 'city', 'state','phone1','gst_no')
            ->first();

         $pdf = PDF::loadView('reportstemplate.todaysOrder', [
            'orders' => $orders,
            'company'=>$company,
                'is_pdf' => true 
            ]);
        return $pdf->download('todays_order.pdf'); 
    }


    // public function inwardInvoice(Request $request){

    //     // Safely handle large data
    //     ini_set('memory_limit', '1024M'); // Increase memory limit
    //     set_time_limit(300);              // Increase max execution time to 5 minutes

    //     $inward_invoice = DB::table('inward_invoice')->where('inward_id', $request->id)->first();
    //     $inward = DB::table('inventory')->find($request->id);
    //     $company = DB::table('companies')->find($inward->company_id);
    //     $po = DB::table('purchase_orders')->where('po_number', $inward_invoice->po_no)->first();
    //     $po_items = DB::table('purchase_order_items')
    //     ->leftJoin('product_variant', 'product_variant.id', '=', 'purchase_order_items.varient_id')
    //     ->where('purchase_order_id', $po->id)
    //     ->select('purchase_order_items.*', 'product_variant.varient_name')
    //     ->get();

    //     $supplier = DB::table('suppliers')->find($po->supplier_id);

    //     $pdf = PDF::loadView('admin.inward_invoice', compact('inward_invoice', 'inward', 'company', 'supplier', 'po', 'po_items'));

    //     return $pdf->download('inward_invoice.pdf');
    //     // return $supplier;
    //     // return view('admin.inward_invoice', compact('inward_invoice', 'inward', 'company', 'supplier', 'po', 'po_items'));
    // }

   public function inwardInvoice(Request $request){

       // dd($request->all());

        // Safely handle large data
        ini_set('memory_limit', '1024M'); // Increase memory limit
        set_time_limit(300);              // Increase max execution time to 5 minutes

        $inward_invoice = DB::table('inward_invoice')->where('inward_id', $request->id)->first();
        //dd( $inward_invoice);
        $inward = DB::table('estimated_po')->where('inward_id',$request->id)->first();
       // dd($inward);
        $inventory = DB::table('inventory')->where('id',$request->id)->first();
        $company = DB::table('companies')->find($inventory->company_id);
       
        $po = DB::table('purchase_orders')->where('id', $inward_invoice->po_number)->first();
       //dd($inward->po_detail_id);
        $po_items = DB::table('purchase_order_items')
        ->leftJoin('product_variant', 'product_variant.id', '=', 'purchase_order_items.varient_id')
         ->leftJoin('products', 'product_variant.product_id', '=', 'products.id')
          ->leftJoin('units', 'units.id', '=', 'purchase_order_items.uom')
        ->where('purchase_order_items.id', $inward->po_detail_id)
        ->select('purchase_order_items.*', 'product_variant.varient_name','product_variant.varient_code','products.product_name','units.unit_name')
        ->get();
      

        $supplier = DB::table('suppliers')->find($po->supplier_id);

        $pdf = PDF::loadView('admin.inward_invoice', compact('inward_invoice', 'inward', 'company', 'supplier', 'po', 'po_items'));

          return $pdf->download('inward_invoice.pdf');
        // return $supplier;
        // return view('admin.inward_invoice', compact('inward_invoice', 'inward', 'company', 'supplier', 'po', 'po_items'));
    }

    public function checkInvoice(Request $request){
        $id = $request->id;
        $checkinvoice = DB::table('inward_invoice')->where('inward_id',$id)->first(); 
        if($checkinvoice){
            $invoicestatus=1;
        }else{
            $invoicestatus=0; 
        }
        return $invoicestatus;
    }

    public function generateInvoice(Request $request){
        $id = $request->id;
        $inward = DB::table('inventory')->find($id); 

        $lastCompany = DB::table('purchase_orders')->orderBy('po_number', 'desc')->where('deleted_at',null)->first();

        if ($lastCompany) {
            $lastIdNumber = (int) substr($lastCompany->po_number, 2); 
            $newPOId = 'PO' . str_pad($lastIdNumber, 6, '0', STR_PAD_LEFT); 
        } else {
            $newPOId = 'PO000001';
        }
        // Get the current timestamp
        $currentTimestamp = Carbon::now();

        // Generate invoice number with timestamp format
        $newInvoiceNumber = 'INV' . $currentTimestamp->format('YmdHis');
        // return $newInvoiceNumber;
        // Insert new invoice into database
        $inward = DB::table('inward_invoice')->insert([
            'invoice_id' => $newInvoiceNumber,
            'inward_id' => $id,
            'po_number' => $inward->po_number, 
            'created_at' => now(),
            'updated_at' => now(),
            'po_no'      => $newPOId
        ]);

        return $inward;
        
    }


    public function getProductsBySupplier($supplierId)
    {
        $products = DB::table('inventory')
            ->join('products', 'products.id', '=', 'inventory.product_id')
            ->where('inventory.supplier_id', $supplierId)
            ->whereNull('products.deleted_at')
            ->select('products.id', 'products.product_name')
            ->groupBy('products.id', 'products.product_name')
            ->get();
        // dd( $products);

        return response()->json($products);
    }

        public function getWarehousesByProduct($productId, $supplierId = null)
        {
            //dd($supplierId);
            $query = DB::table('inventory')
                ->join('rolls', 'inventory.id', '=', 'rolls.inward_id')
                ->join('product_variant', 'product_variant.id', '=', 'rolls.varient_id')
                ->join('products', 'products.id', '=', 'product_variant.product_id')
                ->join('warehouses', 'warehouses.id', '=', 'inventory.warehouse_id')
                ->where('products.id', $productId);

            if ($supplierId) {
                $query->where('inventory.supplier_id', $supplierId);
            }

            $warehouses = $query->select('warehouses.id', 'warehouses.warehouse_name')
                ->groupBy('warehouses.id', 'warehouses.warehouse_name')
                ->get();

            return response()->json($warehouses);
        }

        
}
